@extends('admin_dashboard')

@section('admin')
    <div class="content">
        <h5>POS Screen</h5>

        @php
            // Fetch any open register for the current user (ignore date)
            $currentRegister = \App\Models\Register::where('user_id', auth()->id())
                ->where('status', 'open')
                ->first();
        @endphp

        @if($currentRegister)
            <!-- VIEW SUMMARY BUTTON -->
            <div class="mb-3 text-center">
                <a href="{{ route('register.summary', $currentRegister->id) }}"
                   class="btn btn-danger btn-lg fw-bold">
                    👁️ View Register Summary
                </a>
            </div>
        @endif

        @if(!$currentRegister)
            <div class="alert alert-danger">
                You must open a register first.
                <a href="{{ route('register.open') }}" class="btn btn-primary btn-sm">Open Register</a>
            </div>
        @else
            <!-- POS RECIPES -->
            <div class="row mb-3">
                @foreach($recipes as $recipe)
                    <div class="col-md-3 mb-3">
                        <button class="btn btn-outline-primary w-100 pos-item"
                                data-id="{{ $recipe['recipe_id'] }}"
                                data-name="{{ $recipe['recipe_name'] }}"
                                data-price="{{ $recipe['selling_price'] }}"
                                data-available="{{ $recipe['available_qty'] }}">
                            {{ $recipe['recipe_name'] }}<br>
                            <small>Available: {{ $recipe['available_qty'] }}</small><br>
                            <small>Price: {{ number_format($recipe['selling_price'], 2) }}</small>
                        </button>
                    </div>
                @endforeach
            </div>

            <hr>

            <!-- CURRENT SALE TABLE -->
            <h5>Current Sale</h5>
            <table class="table table-bordered" id="saleTable">
                <thead>
                <tr>
                    <th>Recipe</th>
                    <th>Qty</th>
                    <th>Unit Price</th>
                    <th>Total</th>
                    <th>Action</th>
                </tr>
                </thead>
                <tbody></tbody>
                <tfoot>
                <tr>
                    <th colspan="3" class="text-end">Grand Total</th>
                    <th id="grandTotal">0.00</th>
                    <th></th>
                </tr>
                </tfoot>
            </table>

            <button class="btn btn-success" id="completeSale">Complete Sale</button>
        @endif
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            let saleItems = {};
            const saleTableBody = document.querySelector('#saleTable tbody');

            // ADD RECIPE TO SALE
            document.querySelectorAll('.pos-item').forEach(btn => {
                btn.addEventListener('click', () => {
                    const id = btn.dataset.id;
                    const name = btn.dataset.name;
                    const price = parseFloat(btn.dataset.price);
                    const available = parseFloat(btn.dataset.available);

                    if (!saleItems[id]) saleItems[id] = { id, name, price, qty: 0, available };

                    if (saleItems[id].qty + 1 > available) {
                        return alert('Not enough stock! Create new kitchen order.');
                    }

                    saleItems[id].qty += 1;
                    renderSaleTable();
                });
            });

            // RENDER SALE TABLE
            function renderSaleTable() {
                saleTableBody.innerHTML = '';
                let grandTotal = 0;

                Object.values(saleItems).forEach(item => {
                    const total = item.qty * item.price;
                    grandTotal += total;

                    const row = document.createElement('tr');
                    row.dataset.id = item.id;
                    row.innerHTML = `
                <td>${item.name}</td>
                <td>${item.qty}</td>
                <td>${item.price.toFixed(2)}</td>
                <td>${total.toFixed(2)}</td>
                <td>
                    <button class="btn btn-sm btn-danger remove-item">X</button>
                </td>
            `;
                    saleTableBody.appendChild(row);
                });

                document.getElementById('grandTotal').innerText = grandTotal.toFixed(2);
            }

            // REMOVE ITEM
            saleTableBody.addEventListener('click', function(e) {
                if (e.target.classList.contains('remove-item')) {
                    const row = e.target.closest('tr');
                    const id = row.dataset.id;
                    delete saleItems[id];
                    renderSaleTable();
                }
            });

            // COMPLETE SALE AJAX
            const completeBtn = document.getElementById('completeSale');
            if (completeBtn) {
                completeBtn.addEventListener('click', () => {
                    if (Object.keys(saleItems).length === 0) return alert('No items in sale');

                    fetch("{{ route('pos.complete') }}", {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': '{{ csrf_token() }}'
                        },
                        body: JSON.stringify({ saleItems })
                    })
                        .then(res => res.json())
                        .then(data => {
                            if (data.status === 'success') {
                                alert('Sale completed successfully!');
                                saleItems = {};
                                renderSaleTable();
                                window.location.reload();
                            } else {
                                alert(data.message || 'Error completing sale');
                            }
                        })
                        .catch(err => {
                            console.error(err);
                            alert('AJAX error. Check console.');
                        });
                });
            }
        });
    </script>
@endsection
