@extends('admin_dashboard')
@section('admin')

    <div class="content">
        <div class="card">
            <div class="card-body">
                <h5>Chef Order Sheet</h5>

                <div class="row mb-3">
                    <div class="col-md-4">
                        <select id="recipeSelect" class="form-select">
                            <option value="">Select Recipe</option>
                            @foreach($recipes as $recipe)
                                <option value="{{ $recipe->id }}">{{ $recipe->recipe_name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-3">
                        <input type="number" id="cookQty" class="form-control" placeholder="Qty to cook">
                    </div>

                    <div class="col-md-2">
                        <button id="addRecipe" class="btn btn-primary">Add</button>
                    </div>
                </div>

                <h6>Recipes Added</h6>
                <table class="table table-sm">
                    <thead>
                    <tr>
                        <th>Recipe</th>
                        <th>Qty</th>
                        <th></th>
                    </tr>
                    </thead>
                    <tbody id="recipeTable"></tbody>
                </table>

                <h6 class="mt-4">Ingredient Summary</h6>
                <table class="table table-bordered">
                    <thead>
                    <tr>
                        <th>Ingredient</th>
                        <th>Required Qty</th>
                        <th>Total Cost</th>
                    </tr>
                    </thead>
                    <tbody id="ingredientTable"></tbody>
                </table>

                <div class="row mt-3">
                    <div class="col-md-6 offset-md-6">
                        <table class="table table-bordered">
                            <tr>
                                <th class="text-end">Grand Total Cost</th>
                                <th class="text-end text-success" id="grandTotal">0.00</th>
                            </tr>
                        </table>
                    </div>
                </div>

                <div class="row mt-3">
                    <div class="col-md-12 text-end">
                        <textarea id="remarks" class="form-control mb-2" placeholder="Enter remarks"></textarea>
                        <button id="saveDraft" class="btn btn-success">Save as Draft</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        let recipesAdded = {};
        let ingredientPool = {};
        let baseStock = {};

        const recipeSelect = document.getElementById('recipeSelect');
        const cookQty = document.getElementById('cookQty');

        // Add recipe
        document.getElementById('addRecipe').onclick = async () => {
            const recipeId = recipeSelect.value;
            const qty = parseFloat(cookQty.value);

            if (!recipeId || qty <= 0) return alert('Enter valid data');

            // Prevent duplicate recipes
            if (recipesAdded[recipeId]) return alert('This recipe is already added!');

            // Fetch ingredients for the quantity
            const res = await fetch(`/chef/recipe/${recipeId}/ingredients?qty=${qty}`);
            const data = await res.json();

            // Add recipe to collection
            recipesAdded[recipeId] = {
                recipeId,
                recipeName: recipeSelect.options[recipeSelect.selectedIndex].text,
                qty,
                ingredients: data.ingredients.map(i => ({
                    ...i,
                    recipe_id: recipeId,
                    recipe_name: recipeSelect.options[recipeSelect.selectedIndex].text,
                    recipe_qty: qty
                }))
            };

            rebuild();
        };

        // Rebuild tables
        function rebuild() {
            ingredientPool = {};
            baseStock = {};
            document.getElementById('recipeTable').innerHTML = '';

            Object.values(recipesAdded).forEach(recipe => {
                // Recipe table
                document.getElementById('recipeTable').insertAdjacentHTML('beforeend', `
                <tr>
                    <td>${recipe.recipeName}</td>
                    <td>${recipe.qty}</td>
                    <td>
                        <button class="btn btn-sm btn-danger" onclick="removeRecipe('${recipe.recipeId}')">Remove</button>
                    </td>
                </tr>
            `);

                // Aggregate ingredient quantities
                recipe.ingredients.forEach(i => {
                    if (!ingredientPool[i.product_name]) {
                        ingredientPool[i.product_name] = { required: 0, cost: 0 };
                        baseStock[i.product_name] = i.stock_available;
                    }
                    ingredientPool[i.product_name].required += i.required_qty;
                    ingredientPool[i.product_name].cost += i.ingredient_cost;
                });
            });

            renderIngredients();
        }

        // Remove recipe
        function removeRecipe(recipeId) {
            delete recipesAdded[recipeId];
            rebuild();
        }

        // Render ingredient summary
        function renderIngredients() {
            const tbody = document.getElementById('ingredientTable');
            const grandTotalEl = document.getElementById('grandTotal');
            tbody.innerHTML = '';
            let grandTotal = 0;

            Object.entries(ingredientPool).forEach(([name, data]) => {
                const stock = baseStock[name];
                const remaining = stock - data.required;
                const shortage = Math.max(0, -remaining);
                grandTotal += data.cost;

                tbody.insertAdjacentHTML('beforeend', `
                <tr>
                    <td>${name}</td>
                    <td>${data.required.toFixed(3)}</td>
                    <td>${Math.max(0, remaining).toFixed(3)}</td>
                    <td class="text-danger">${shortage.toFixed(3)}</td>
                    <td>${data.cost.toFixed(2)}</td>
                </tr>
            `);
            });

            grandTotalEl.innerText = grandTotal.toFixed(2);
        }

        // Save draft
        document.getElementById('saveDraft').onclick = async () => {
            if (Object.keys(recipesAdded).length === 0) return alert('No recipes added!');

            const payload = {
                remarks: document.getElementById('remarks')?.value || null,
                recipes: Object.values(recipesAdded).map(r => ({
                    recipe_id: r.recipeId,
                    recipe_name: r.recipeName,
                    recipe_qty: r.qty,
                    ingredients: r.ingredients.map(i => ({
                        product_id: i.product_id,
                        required_qty: i.required_qty,
                        stock_available: i.stock_available,
                        ingredient_unit_cost: i.ingredient_unit_cost,
                        ingredient_cost: i.ingredient_cost,
                        shortage_qty: i.shortage_qty,
                        recipe_id: i.recipe_id,
                        recipe_name: i.recipe_name,
                        recipe_qty: i.recipe_qty
                    }))
                }))
            };

            try {
                const res = await fetch('/chef/kitchen-order/save', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify(payload)
                });

                const data = await res.json();
                if (data.status === 'success') {
                    alert(data.message);
                    recipesAdded = {};
                    rebuild();
                    document.getElementById('remarks').value = '';
                } else {
                    alert('Error saving order');
                }

            } catch (err) {
                console.error(err);
                alert('Error saving order');
            }
        };
    </script>

@endsection
