<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Kitchen_order_approval;
use App\Models\Kitchen_order_ingredient;
use App\Models\Register;
use App\Models\Register_closure;
use App\Models\Register_closure_item;
use App\Models\Sale;
use App\Models\Sale_item;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RegisterController extends Controller
{
// GET - Show the open register form
    public function openForm()
    {
        return view('backend.pos.open_register'); // <-- show the blade
    }

// POST - Submit opening cash

    public function open(Request $request)
    {
        $request->validate([
            'opening_cash' => 'required|numeric|min:0',
        ]);

        // 🔍 Check for ANY open register (ignore date)
        $existingRegister = Register::where('user_id', auth()->id())
            ->where('status', 'open')
            ->first();

        // ✅ If already open → go to POS
        if ($existingRegister) {
            return redirect()->route('pos.index')
                ->with('info', 'You already have an open register. Please close it before opening a new one.');
        }

        // 🆕 Create new register (TODAY)
        Register::create([
            'user_id'       => auth()->id(),
            'business_date' => now()->toDateString(),
            'opening_cash'  => $request->opening_cash,
            'opened_at'     => now(),
            'status'        => 'open',
        ]);

        return redirect()->route('pos.index')
            ->with('success', 'Register opened successfully.');
    }


    public function closeRegister(Request $request, Register $register)
{
//        dd($register->id, $register->status, auth()->id());
    // 🔐 Security
    if ($register->user_id !== auth()->id()) {
        abort(403, 'Unauthorized');
    }

    if ($register->status !== 'open') {
        return back()->with('error', 'Register already closed');
    }

    DB::beginTransaction();

    try {

        // 1️⃣ SALES SUMMARY
        $sales = Sale::where('register_id', $register->id)->get();

        $totalSales        = $sales->sum('total_amount');
        $totalTransactions = $sales->count();
        $totalItemsSold    = Sale_item::whereIn('sale_id', $sales->pluck('id'))->sum('qty');

        // 2️⃣ CREATE REGISTER CLOSURE
        $closure = Register_closure::create([
            'register_id'        => $register->id,
            'user_id'            => auth()->id(),
            'total_sales'        => $totalSales,
            'total_transactions' => $totalTransactions,
            'total_items_sold'   => $totalItemsSold,
            'expected_cash'      => $totalSales,
            'actual_cash'        => $request->actual_cash ?? $totalSales,
            'variance'           => ($request->actual_cash ?? $totalSales) - $totalSales,
            'closed_at'          => now(),
        ]);

        $approved = Kitchen_order_approval::where('register_id', $register->id)
            ->selectRaw('recipe_id, recipe_name, SUM(approved_qty) as approved_qty')
            ->groupBy('recipe_id', 'recipe_name')
            ->get()
            ->keyBy('recipe_id');

        $sold = Sale_item::whereIn('sale_id', $sales->pluck('id'))
            ->selectRaw('recipe_id, SUM(qty) as sold_qty')
            ->groupBy('recipe_id')
            ->get()
            ->keyBy('recipe_id');

        foreach ($approved as $recipeId => $data) {

            $soldQty = $sold[$recipeId]->sold_qty ?? 0;

            Register_closure_item::create([
                'register_closure_id' => $closure->id,
                'recipe_id'           => $recipeId,
                'recipe_name'         => $data->recipe_name,
                'approved_qty'        => $data->approved_qty,   // ✅ ONLY FROM APPROVALS
                'sold_qty'            => $soldQty,
                'remaining_qty'       => $data->approved_qty - $soldQty,
            ]);
        }

        // 4️⃣ CLOSE REGISTER
        $register->update([
            'status'    => 'closed',
            'closed_at' => now(),
        ]);



        DB::commit();

        return redirect()->route('register.summary', $register->id)
            ->with('success', 'Register closed successfully!');

    } catch (\Throwable $e) {
        DB::rollBack();
        return back()->with('error', $e->getMessage());
    }
}



    public function summary(Register $register)
    {
        // 🔐 Security check
        if ($register->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        if ($register->status !== 'open') {
            return redirect()->route('pos.index')
                ->with('error', 'Register is not open.');
        }

        // Sales linked to this register
        $sales = $register->sales()->with('items')->get();

        $totalSales        = $sales->sum('total_amount');
        $totalTransactions = $sales->count();
        $totalItemsSold    = $sales->pluck('items')->flatten()->sum('qty');

        // Approved kitchen production linked to this register
        $approved = $register->kitchenApprovals()
            ->selectRaw('recipe_id, recipe_name, SUM(approved_qty) as approved_qty')
            ->groupBy('recipe_id', 'recipe_name')
            ->get()
            ->keyBy('recipe_id');

        // Sold quantities
        $sold = Sale_item::whereIn('sale_id', $sales->pluck('id'))
            ->selectRaw('recipe_id, SUM(qty) as sold_qty')
            ->groupBy('recipe_id')
            ->get()
            ->keyBy('recipe_id');

        // Combine kitchen approvals vs sales
        $items = [];
        foreach ($approved as $recipeId => $data) {
            $soldQty = $sold[$recipeId]->sold_qty ?? 0;

            $items[] = (object)[
                'recipe_name'   => $data->recipe_name,
                'approved_qty'  => (int) $data->approved_qty,
                'sold_qty'      => (int) $soldQty,
                'remaining_qty' => max(0, $data->approved_qty - $soldQty),
            ];
        }

        // Cash reconciliation placeholders
        $expectedCash = $totalSales;
        $actualCash   = null;
        $variance     = $actualCash !== null ? $actualCash - $expectedCash : 0;

        return view('backend.pos.register_summary', compact(
            'register',
            'items',
            'totalSales',
            'totalTransactions',
            'totalItemsSold',
            'expectedCash',
            'actualCash',
            'variance'
        ));
    }


    public function allRegisters()
    {
        $closures = Register_closure::with([
            'register',
            'user',
            'items'
        ])
            ->orderBy('closed_at', 'desc')
            ->get();

        return view('backend.pos.all_registers', compact('closures'));
    }









}
