<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Kitchen_order;
use App\Models\Kitchen_order_approval;
use App\Models\Kitchen_order_ingredient;
use App\Models\Meal;
use App\Models\Recipe;
use App\Models\Register;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class KitchenController extends Controller
{
    //
    public function AddOrder()
    {  $recipes = Recipe::all();
        return view('backend.kitchen.kitchen_order', compact('recipes'));
    }
    public function checkRecipeIngredients(Request $request, $recipeId)
    {
        // Get quantity to cook from query, default to 1 if invalid
        $qtyToCook = max(1, (float) $request->query('qty', 1));

        // Load recipe with its ingredients and products
        $recipe = Recipe::with('ingredients.product')->find($recipeId);
        if (!$recipe) {
            return response()->json([
                'recipe_yield' => 0,
                'cook_qty' => $qtyToCook,
                'ingredients' => []
            ]);
        }

        // Map ingredients to include all required info
        $ingredients = $recipe->ingredients->map(function ($ingredient) use ($qtyToCook, $recipe) {

            $product = $ingredient->product;
            if (!$product) return null; // skip missing product

            // Calculate required quantity based on requested cook qty
            $requiredQty = ($qtyToCook / $recipe->yield_qty) * $ingredient->quantity;

            // Ingredient cost for this quantity
            $ingredientCost = $requiredQty * $ingredient->unit_cost;

            // Stock check
            $stockAvailable = $product->stock ?? 0;
            $shortageQty = max(0, $requiredQty - $stockAvailable);

            return [
                'product_id'          => $product->id,
                'product_name'        => $product->product_name,
                'required_qty'        => round($requiredQty, 3),
                'stock_available'     => round($stockAvailable, 3),
                'shortage_qty'        => round($shortageQty, 3),
                'ingredient_unit_cost'=> round($ingredient->unit_cost, 2),
                'ingredient_cost'     => round($ingredientCost, 2),
            ];
        })
            ->filter() // remove nulls if product missing
            ->values(); // reindex array

        return response()->json([
            'recipe_yield' => $recipe->yield_qty,
            'cook_qty'     => $qtyToCook,
            'ingredients'  => $ingredients
        ]);
    }
    // Return ingredients of a recipe with calculated qty
    public function recipeIngredients(Recipe $recipe, Request $request)
    {
        $qty = $request->query('qty', 1);

        $ingredients = $recipe->ingredients->map(function($ing) use ($qty) {
            return [
                'product_id' => $ing->product_id,
                'product_name' => $ing->product->name,
                'required_qty' => $ing->pivot->required_qty * $qty,
                'unit_cost' => $ing->unit_cost,
                'ingredient_cost' => $ing->unit_cost * $ing->pivot->required_qty * $qty,
                'stock_available' => $ing->stock_available ?? 0,
            ];
        });

        return response()->json(['ingredients' => $ingredients]);
    }


    private function storeIngredients($order, $ingredients, $sourceName, $orderQty, $sourceType, $recipeId)
    {
        $subTotal = 0;
        foreach ($ingredients as $ing) {
            // $ing is an instance of RecipeIngredients model
            $requiredQtyTotal = $ing->required_qty * $orderQty;
            $unitCost = $ing->product->unit_cost ?? 0; // Assuming unit_cost is on Product
            $totalCost = $requiredQtyTotal * $unitCost;

            Kitchen_order_ingredient::create([
                'kitchen_order_id' => $order->id,
                'product_id'       => $ing->product_id,
                'required_qty'     => $requiredQtyTotal,
                'recipe_name'      => $sourceName,
                'source_type'      => $sourceType,
                'recipe_qty'       => $orderQty,
                'recipe_id'        => $recipeId,
                'unit_cost'        => $unitCost,
                'total_cost'       => $totalCost,
                'stock_available'  => $ing->product->stock_available ?? 0,
                'shortage_qty'     => max(0, $requiredQtyTotal - ($ing->product->stock_available ?? 0)),
            ]);

            $subTotal += $totalCost;
        }
        return $subTotal;
    }



    // 1. View all drafts for the current chef
    public function viewDrafts()
    {
        $orders = Kitchen_order::latest()->get();


        return view('backend.kitchen.kitchen_drafts', compact('orders'));
    }
    // 2. Preview a specific order
    public function previewOrder(Kitchen_order $order)
    {
//        $this->authorize('view', $order); // optional policy for security

        $order->load('ingredients.product');

        return view('backend.kitchen.kitchen_preview_order', compact('order'));
    }

    // 3. Approve the order
    public function approveOrder(Request $request, Kitchen_order $order)
    {
        DB::transaction(function () use ($order) {

            foreach ($order->ingredients as $ingredient) {

                $product = $ingredient->product;

                // Deduct stock even if it becomes negative
                if ($product) {
                    $product->update([
                        'stock' => $product->stock - $ingredient->required_qty
                    ]);
                }

                // Update kitchen_order_ingredients (NO register, NO stock checks)
                $ingredient->update([
                    'recipe_qty'      => $ingredient->recipe_qty,
                    'required_qty'    => $ingredient->required_qty,
                    'stock_available' => $product?->stock ?? 0,
                    'shortage_qty'    => 0, // no blocking, no shortage logic
                ]);

                // Log approval
                $totalCost = $ingredient->required_qty * $ingredient->unit_cost;

                Kitchen_order_approval::create([
                    'kitchen_order_id' => $order->id,
                    'recipe_id'        => $ingredient->recipe_id,
                    'recipe_name'      => $ingredient->recipe_name,
                    'approved_qty'     => $ingredient->recipe_qty,
                    'unit_cost'        => $ingredient->unit_cost,
                    'total_cost'       => $totalCost,
                    'user_id'          => auth()->id(),
                    'order_date'       => now(),
                    // ❌ no register_id
                ]);
            }

            // Final approval
            $order->update([
                'status' => 'approved'
            ]);
        });

        return redirect()
            ->route('chef.kitchen.drafts')
            ->with('success', 'Order approved successfully.');
    }


    public function mealIngredients(Meal $meal, Request $request)
    {
        $qty = (float) $request->qty;

        $meal->load('recipes.ingredients.product');

        $ingredients = [];

        foreach ($meal->recipes as $recipe) {
            foreach ($recipe->ingredients as $ingredient) {

                $requiredQty = $ingredient->quantity * $qty;

                $ingredients[] = [
                    'product_id' => $ingredient->product_id,
                    'product_name' => $ingredient->product->product_name,
                    'required_qty' => $requiredQty,
                    'ingredient_cost' => $requiredQty * $ingredient->product->unit_cost,
                ];
            }
        }

        return response()->json([
            'ingredients' => $ingredients
        ]);
    }

















    // 4. Generate PDF for printing
    public function downloadPdf(Kitchen_order $order)
    {
        // Load relationships
        $order->load('ingredients.product.unit');

        // 🔹 Group ingredients across all recipes
        $groupedIngredients = $order->ingredients
            ->groupBy('product_id')
            ->map(function ($items) {
                return [
                    'ingredient' => $items->first()->product->product_name ?? 'N/A',
                    'unit'       => $items->first()->product->unit->unit_symbol ?? '',
                    'total_qty'  => $items->sum('required_qty'),
                ];
            });

        // 🔹 Pass BOTH variables to the view
        $pdf = PDF::loadView(
            'backend.kitchen.kitchen_oder_pdf',
            compact('order', 'groupedIngredients')
        );

        return $pdf->download('KitchenOrder_' . $order->order_no . '.pdf');
    }

    public function saveKitchenOrder(Request $request)
    {
        $request->validate([
            'recipes' => 'required|array|min:1',
        ]);

        DB::transaction(function () use ($request) {
            $order = Kitchen_order::create([
                'order_no' => 'KO-' . now()->format('YmdHis'),
                'order_date' => now()->toDateString(),
                'chef_id' => Auth::id(), // automatically use logged-in user
                'status' => 'draft',
                'total_cost' => 0,
                'remarks' => $request->remarks,
            ]);

            $grandTotal = 0;

            foreach ($request->recipes as $recipe) {
                foreach ($recipe['ingredients'] as $ing) {
                    Kitchen_order_ingredient::create([
                        'kitchen_order_id' => $order->id,
                        'product_id' => $ing['product_id'],
                        'required_qty' => $ing['required_qty'],
                        'unit_cost' => $ing['ingredient_unit_cost'],
                        'recipe_id' => $ing['recipe_id'],          // ✔ this is key
                        'recipe_name' => $ing['recipe_name'],      // optional for display
                        'recipe_qty' => $ing['recipe_qty'],
                        'total_cost' => $ing['ingredient_cost'],
                        'stock_available' => $ing['stock_available'],
                        'shortage_qty' => $ing['shortage_qty'],
                    ]);

                    $grandTotal += $ing['ingredient_cost'];
                }
            }

            $order->update(['total_cost' => $grandTotal]);
        });

        return response()->json(['status' => 'success', 'message' => 'Kitchen order saved as draft']);
    }





}
