<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Allowance;
use App\Models\Deduction;
use App\Models\Department;
use App\Models\Designation;
use App\Models\Employee;
use Illuminate\Http\Request;
use App\Exports\EmployeesExport;
use App\Imports\EmployeesImport;
use Maatwebsite\Excel\Facades\Excel;


class EmployeeController extends Controller
{
    //
    public function index()
    {
        return view('backend.employees.employees_index', [
            'employees' => Employee::with(['department','designation','allowances','deductions'])->latest()->get(),
            'departments' => Department::where('status',1)->get(),
            'designations' => Designation::where('status',1)->get(),
        ]);
    }


    public function store(Request $request)
    {
        $request->validate([
            'full_name' => 'required',
            'department_id' => 'required',
            'designation_id' => 'required',
            'basic_salary' => 'required|numeric',
            'hire_date' => 'required|date',
        ]);

        // Auto-generate employee code
        $lastEmployee = Employee::latest()->first();
        $nextNumber = $lastEmployee ? $lastEmployee->id + 1 : 1;
        $employeeCode = 'EMP-' . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);

        Employee::create([
            'employee_code' => $employeeCode,
            'full_name' => $request->full_name,
            'department_id' => $request->department_id,
            'designation_id' => $request->designation_id,
            'gender' => $request->gender,
            'basic_salary' => $request->basic_salary,
            'hire_date' => $request->hire_date,
        ]);
        $notification = [
            'message' => 'Employee Added Successfully',
            'alert-type' => 'success',
        ];


        return back()->with($notification);
    }
    public function update(Request $request, $id)
    {
        $request->validate([
            'full_name' => 'required',
            'department_id' => 'required',
            'designation_id' => 'required',
            'gender' => 'required',
            'basic_salary' => 'required|numeric',
            'hire_date' => 'required|date',
        ]);

        $employee = Employee::findOrFail($id);

        $employee->update([
            'full_name' => $request->full_name,
            'department_id' => $request->department_id,
            'designation_id' => $request->designation_id,
            'basic_salary' => $request->basic_salary,
            'gender' => $request->gender,
            'hire_date' => $request->hire_date,
            'status' => $request->status ?? $employee->status,
        ]);
        $notification = [
            'message' => 'Employee Data Updated Successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }
    public function export()
    {
        return Excel::download(new EmployeesExport, 'employees.xlsx');
    }
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv',
        ]);

        Excel::import(new EmployeesImport, $request->file('file'));
        $notification = [
            'message' => 'Employee Data Updated Successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }
    public function editPayroll($id)
    {
        $employee = Employee::with(['allowances','deductions'])->findOrFail($id);
        $allowances = Allowance::all();
        $deductions = Deduction::all();

        return view('backend.employees.edit_payroll', compact('employee','allowances','deductions'));
    }
    public function updatePayroll(Request $request, $id)
    {
        $employee = Employee::findOrFail($id);

        // Sync selected allowances
        $employee->allowances()->sync($request->allowances ?? []);

        // Sync selected deductions
        $employee->deductions()->sync($request->deductions ?? []);

        $notification = [
            'message' => ' Updated Successfully',
            'alert-type' => 'success',
        ];
        return redirect()->back()->with($notification);
    }
    public function payrollIndex()
    {
        $employees = Employee::with(['allowances', 'deductions', 'department'])->latest()->get();
        $allowances = Allowance::all();
        $deductions = Deduction::all();

        return view('backend.payroll.payroll_index', compact('employees', 'allowances', 'deductions'));
    }

// Assign allowances & deductions to employee

    public function assignPayroll(Request $request, $id)
    {
        $employee = Employee::findOrFail($id);

        // Clear previous assignments first (optional)
        $employee->allowances()->detach();
        $employee->deductions()->detach();

        // Assign new allowances
        if ($request->has('allowances')) {
            foreach ($request->allowances as $allowance_id) {
                $employee->allowances()->attach($allowance_id, [
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }
        }

        // Assign new deductions
        if ($request->has('deductions')) {
            foreach ($request->deductions as $deduction_id) {
                $employee->deductions()->attach($deduction_id, [
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }
        }
        $notification = [
            'message' => ' Updated Successfully',
            'alert-type' => 'success',
        ];
        return redirect()->route('employees.index')->with($notification, 'Payroll updated successfully for ' . $employee->full_name);
    }
    public function create(Employee $employee)
    {
        return view('backend.employees.assign_benefits', [
            'employee'   => $employee,
            'allowances' => Allowance::all(),
            'deductions' => Deduction::all(),
        ]);
    }
    public function Assignindex()
    {
        $employees = Employee::all();
        return view('backend.employees.assign_benefits', compact('employees'));
    }
    public function showAssignBenefits($id)
    {
        $employee = Employee::with(['allowances', 'deductions'])->findOrFail($id);

        $allowances = Allowance::all();
        $deductions = Deduction::all();

        return view('backend.employees.assign_benefits', compact(
            'employee',
            'allowances',
            'deductions'
        ));
    }


}
